﻿import GC from "@grapecity/spread-sheets";
import "@grapecity/spread-sheets-print";
import "@grapecity/spread-sheets-shapes";
import "@grapecity/spread-sheets-pivot-addon";
import "@grapecity/spread-sheets-tablesheet";
import "@grapecity/spread-sheets-io";
import '@grapecity/spread-sheets-designer-resources-en';
import '@grapecity/spread-sheets-designer';
import '@grapecity/spread-sheets-designer/styles/gc.spread.sheets.designer.min.css';
import { SyncFunction } from './syncFormula';

GC.Spread.Common.CultureManager.culture("en-us");

const designer = new (GC.Spread.Sheets as any).Designer.Designer(document.getElementById('app'));
const spread = designer.getWorkbook();

// For Debugging
(window as any)["GC"] = GC;
(window as any)["spread"] = spread;

const functionNames = {
  asyncFunc: "ASYNC",
  syncFunc: "CUSTOMSYNCFORMULA",
};

GC.Spread.CalcEngine.Functions.defineGlobalCustomFunction(functionNames.syncFunc, new SyncFunction());

type FunctionExpressionType = {
  functionName: string;
  arguments: GC.Spread.CalcEngine.Expression[];
} & GC.Spread.CalcEngine.Expression;

function containsCustomFormula(node: any) {
  const functionName = node.functionName;
  if (functionName && Object.values(functionNames).includes(functionName)) {
    return true;
  }

  // otherwise we check all the arguments of the current node
  const nodeArguments = node.arguments;
  if (nodeArguments && nodeArguments.length > 0) {
    for (let i = 0; i < nodeArguments.length; i++) {
      if (containsCustomFormula(nodeArguments[i])) {
        return true;
      }
    }
  }

  // if the node is a combination of formulas then we have to
  // verify all the formulas in that node
  // if the node is a combination of formulas is put by spreadjs
  // i two properties (recursively) value and value2
  const value = (node).value;
  let valueIsCustomFormula = false;
  const value2 = (node).value2;
  let value2IsCustomFormula = false;
  if (value) {
    valueIsCustomFormula = containsCustomFormula(value);
  }
  if (value2) {
    value2IsCustomFormula = containsCustomFormula(value2);
  }
  if (valueIsCustomFormula || value2IsCustomFormula) {
    return true;
  }

  // there is no custom function used in the formula
  return false;
}

const serializeForExport = (workbook: GC.Spread.Sheets.Workbook) => {
  let json: any = workbook.toJSON();
  let customFnSharedID: any = {};
  for (let sheetName in json.sheets) {
    let sheet = json.sheets[sheetName];
    if (sheet.sharedFormulas) {
      let sharedFormulas = sheet.sharedFormulas;
      for (let name in sharedFormulas) {
        let node = sharedFormulas[name];
        if (node.formula) {
          const formulaNode = GC.Spread.Sheets.CalcEngine.formulaToExpression(workbook.getActiveSheet(), node.formula);
          if (containsCustomFormula(formulaNode)) {
            customFnSharedID[name] = true;
            delete sharedFormulas[name];
          }
        }
      }
    }
    sheet.tables?.forEach((table: any) => {
      table.columns?.forEach((col: any) => {
        const formula = col.dataAreaFormula;
        if (formula) {
          const expression = GC.Spread.Sheets.CalcEngine.formulaToExpression(sheet, formula);
          if (containsCustomFunction(expression as FunctionExpressionType)) {
            delete col.dataAreaFormula;
          }
        }
      });
    });
  }

  const jsonStr = JSON.stringify(json,
    (key, value) => {
      // if there is a custom function in the formula then we do not
      // want to retun the formula while exporting
      let containsCustomFunc = false;
      if (key === "formula") {
        if (typeof value === "string") {
          const formulaNode = GC.Spread.Sheets.CalcEngine.formulaToExpression(workbook.getActiveSheet(), value);
          containsCustomFunc = containsCustomFormula(formulaNode);
        } else if (value && value.si !== void 0 && customFnSharedID[value.si]) {
          containsCustomFunc = true;
        }
      }
      return key === "formula" && containsCustomFunc ? "" : value;
    }
  );

  return jsonStr;
}

document.getElementById('export').onclick = function () {
  console.log("export");
  let serializedJSON = serializeForExport(spread);
  let newWorkbook = new GC.Spread.Sheets.Workbook();
  newWorkbook.fromJSON(JSON.parse(serializedJSON));
  newWorkbook.export(function (blob: any) {
    // save blob to a file
    saveAs(blob, 'Export.xlsx');
  }, function (e: any) {
    console.log(e);
  }, {
    fileType: GC.Spread.Sheets.FileType.excel
  });
}

const containsCustomFunction = (node: FunctionExpressionType) => {
  // if the functionName is included in the custom
  // functions list already defined then we return true
  const functionName = node?.functionName;
  if (functionName && Object.values(functionNames).includes(functionName)) {
    return true;
  }

  // otherwise we check all the arguments of the current node
  const nodeArguments = node?.arguments;
  if (nodeArguments?.length > 0) {
    for (let i = 0; i < nodeArguments?.length; i++) {
      if (containsCustomFunction(nodeArguments[i] as FunctionExpressionType)) {
        return true;
      }
    }
  }

  // if the node is a combination of formulas then we have to
  // verify all the formulas in that node
  // if the node is a combination of formulas is put by spreadjs
  // in two properties (recursively) value and value2
  const value = (node as any)?.value;
  let valueIsNecFormula = false;
  const value2 = (node as any)?.value2;
  let value2IsNecFormula = false;
  if (value) {
    valueIsNecFormula = containsCustomFunction(value);
  }
  if (value2) {
    value2IsNecFormula = containsCustomFunction(value2);
  }

  if (valueIsNecFormula || value2IsNecFormula) {
    return true;
  }

  // there is no custom function used in the formula
  return false;
};
