﻿Imports System
Imports System.ComponentModel
Imports System.Reflection
Imports System.Resources

<AttributeUsage(AttributeTargets.Property Or AttributeTargets.Field)>
Public Class LocalizedDisplayNameAttribute
    Inherits DisplayNameAttribute

    Private ReadOnly _resourceType As Type
    Private ReadOnly _resourceKey As String

    Public Sub New(resourceKey As String, resourceType As Type)
        MyBase.New(resourceKey)
        _resourceType = resourceType
        _resourceKey = resourceKey
    End Sub

    Public Overrides ReadOnly Property DisplayName As String
        Get
            If _resourceType Is Nothing OrElse String.IsNullOrEmpty(_resourceKey) Then
                Return MyBase.DisplayName
            End If

            Try
                ' Attempt 1 — get the ResourceManager directly via reflection (if available)
                Dim rmProp = _resourceType.GetProperty("ResourceManager", BindingFlags.Public Or BindingFlags.NonPublic Or BindingFlags.Static)
                If rmProp IsNot Nothing Then
                    Dim rm = TryCast(rmProp.GetValue(Nothing, Nothing), ResourceManager)
                    If rm IsNot Nothing Then
                        Dim val = rm.GetString(_resourceKey)
                        If Not String.IsNullOrEmpty(val) Then Return val
                    End If
                End If

                ' Attempt 2 — fallback: build ResourceManager dynamically
                ' This works even if resource file isn’t in "My Project" or "Properties"
                Dim asm = _resourceType.Assembly
                Dim asmName = asm.GetName().Name

                ' Determine possible resource base names dynamically
                Dim possibleNames = New List(Of String) From {
                    $"{asmName}.Resources",
                    $"{asmName}.Properties.Resources",
                    $"{asmName}.My.Resources.Resources"
                }

                For Each baseName In possibleNames
                    Try
                        Dim rm As New ResourceManager(baseName, asm)
                        Dim value = rm.GetString(_resourceKey)
                        If Not String.IsNullOrEmpty(value) Then
                            Return value
                        End If
                    Catch
                        ' continue
                    End Try
                Next
            Catch ex As Exception
                ' ignore and fallback
                Debug.WriteLine($"{ex.Message}\n{ex.StackTrace}\n")
            End Try

            Return MyBase.DisplayName
        End Get
    End Property
End Class
