import * as GC from '@mescius/spread-sheets';
import '@mescius/spread-sheets/styles/gc.spread.sheets.excel2013white.css';
import '@mescius/spread-sheets-collaboration-addon';
import { Client } from '@mescius/js-collaboration-client';
import * as OT from '@mescius/js-collaboration-ot-client';
import { Presence } from '@mescius/js-collaboration-presence-client';
import { type, bind, bindPresence } from '@mescius/spread-sheets-collaboration-client';

OT.TypesManager.register(type);

const SERVER_URL = 'ws://localhost:8080';
const ROOM_ID = 'spreadjs-demo-room';

const USER_ID = 'user_' + Date.now();
const USER_NAME = 'User' + Math.floor(Math.random() * 1000);
const USER_COLORS = ['#5B8FF9', '#5AD8A6', '#F6BD16', '#6F5EF9', '#FF6B6B', '#9254DE'];
const USER_COLOR = USER_COLORS[Math.floor(Math.random() * USER_COLORS.length)];

const statusEl = document.getElementById('status');
const userInfoEl = document.getElementById('userInfo');
const presenceListEl = document.getElementById('presenceList');

function updateStatus(connected) {
    statusEl.textContent = connected ? 'Connected' : 'Disconnected';
    statusEl.className = connected ? 'connected' : 'disconnected';
}

function updatePresenceUI(presences) {
    if (!presences || presences.length === 0) {
        presenceListEl.innerHTML = '<span style="color: rgba(255,255,255,0.7); font-size: 12px;">No other users</span>';
        return;
    }

    presenceListEl.innerHTML = '';
    presences.forEach(p => {
        const badge = document.createElement('div');
        badge.className = 'presence-badge';
        badge.innerHTML = `<span class="color-swatch" style="background: ${p.user.color}"></span>${p.user.name}`;
        presenceListEl.appendChild(badge);
    });
}

async function initCollaboration() {
    try {
        const workbook = new GC.Spread.Sheets.Workbook(document.getElementById('app'), { sheetCount: 1 });
        const sheet = workbook.getActiveSheet();
        sheet.name('Collaborative Sheet');

        sheet.setValue(0, 0, 'SpreadJS V19 - Real-Time Collaboration Demo');
        sheet.getRange(0, 0, 1, 6).font('bold 14pt Arial').foreColor('#667eea');

        const headers = ['Product', 'Q1', 'Q2', 'Q3', 'Q4', 'Total'];
        headers.forEach((h, i) => sheet.setValue(2, i, h));

        sheet.getRange(2, 0, 1, 6).backColor('#667eea').foreColor('#fff').font('bold 11pt Arial').hAlign(GC.Spread.Sheets.HorizontalAlign.center);

        const products = ['Laptop', 'Monitor', 'Keyboard', 'Mouse', 'Headset'];
        for (let i = 0; i < products.length; i++) {
            sheet.setValue(3 + i, 0, products[i]);
            for (let j = 1; j <= 4; j++) {
                sheet.setValue(3 + i, j, Math.floor(Math.random() * 10000) + 5000);
            }
        }

        console.log('Workbook initialized');

        const conn = new Client(SERVER_URL).connect(ROOM_ID);

        conn.on('connected', () => {
            console.log('Connected to server');
            updateStatus(true);
        });

        conn.on('disconnected', () => {
            console.log('Disconnected');
            updateStatus(false);
        });

        const doc = new OT.SharedDoc(conn);
        doc.on('error', err => console.error('Error:', err));

        await doc.fetch();

        if (!doc.type) {
            const snapshot = workbook.collaboration.toSnapshot();
            await doc.create(snapshot, type.uri, {});
        }

        bind(workbook, doc);

        const presence = new Presence(conn);
        const user = {
            id: USER_ID,
            name: USER_NAME,
            color: USER_COLOR,
            permission: { mode: 0 }
        };

        userInfoEl.innerHTML = `<span class="color-swatch" style="background: ${USER_COLOR}"></span>You: ${USER_NAME}`;

        await bindPresence(workbook, presence, user, {
            onPresencesUpdate: presences => {
                updatePresenceUI(presences);
            }
        });

        console.log('Setup complete');

    } catch (err) {
        console.error('Failed to initialize:', err);
        statusEl.textContent = 'Connection Failed';
        statusEl.className = 'disconnected';
    }
}

window.addEventListener('load', initCollaboration);
