import { Avatar, Modal, Popconfirm, Space, Table, TableProps, message } from "antd";
import { IDoc, docReq } from "../request/doc";
import { getDocIcon } from "../util/icon";
import { Link, useNavigate } from "react-router-dom";
import { Share } from "./share";
import { useState } from "react";
import { TableRefreshButton } from "./table-refresh-button";
import { Role } from "../enum";
import {translateToLocalDate} from "../util/util";

export function FileList(props: { style?: React.CSSProperties, files: IDoc[], refresh: () => void }) {
    const { files } = props;
    const navigate = useNavigate();
    const [shareDocId, setShareDocId] = useState('');

    const openDoc = (id: string) => {
        navigate(`/open/${id}`);
    };

    const deleteDoc = async (id: string) => {
        console.log('delete doc', id);
        await docReq.delete(id);

        message.success({
            content: 'Delete Successful!',
            duration: 1,
        });

        props.refresh();
    };

    const columns: TableProps<IDoc>['columns'] = [
        {
            title: 'Name',
            dataIndex: 'name',
            key: 'name',
            render: (text, record) => (
                <Space size="large">
                    <Avatar shape='square' src={getDocIcon(record.type)} /><b>{text}</b>
                </Space>
            ),
            onCell: (record) => {
                return {
                    onClick: () => openDoc(record.id),
                };
            },
        },
        {
            title: 'Role',
            dataIndex: 'role',
            key: 'role',
            render: (text) => (
                <p>{text}</p>
            )
        },
        {
            title: 'Last Modify',
            dataIndex: 'lastModify',
            key: 'lastModify',
            render: (_, record) => (<p>{translateToLocalDate(record.lastModifyDate)}</p>)
        },
        {
            title: 'Owner',
            dataIndex: 'createUsername',
            key: 'createUsername',
            render: (_, record) => (
                <p>{record.createUser.username}</p>
            )
        },
        {
            title: 'Action',
            key: 'action',
            render: (_, record) => (
                <Space size="middle">
                    <Link className='action-item' to={`/open/${record.id}`}>Open</Link>
                    <Link className='action-item' to={`/history/${record.id}`}>History</Link>

                    <Popconfirm
                        title="Delete the doc"
                        description="Are you sure to delete this doc?"
                        onConfirm={() => deleteDoc(record.id)}
                        okText="Yes"
                        cancelText="No"
                    >
                        <a>Delete</a>
                    </Popconfirm>

                    {record.role === Role.owner ? <a onClick={() => setShareDocId(record.id)}>Share</a> : null}
                </Space>
            ),
        },
    ];

    return (
        <div style={props.style}>
            <TableRefreshButton refresh={props.refresh} />
            <Table
                dataSource={files}
                columns={columns}
                size="small"
                rowKey={(r) => r.id}
            />
            <Modal destroyOnClose open={shareDocId !== ''} onCancel={() => { setShareDocId('') }} footer={null}>
                <Share docId={shareDocId}></Share>
            </Modal>
        </div>
    );
}

