import {Presence} from "@mescius/js-collaboration-presence-client";

export interface IUser {
    id: string;
    color: string;
    name: string;
}
export interface IPresence {
    user: IUser;
}
interface IDom extends HTMLElement {
    _mouseoverHandler: () => void;
    _mouseoutHandler: () => void;
}

export function bindPresence(presence: Presence<IPresence>, user: IUser, userHostId: string, options?: any) {
    const DEFAULT_COLOR_SCHEME = ['#0000ff', '#008000', '#9900cc', '#800000', '#00cc33', '#cc6600', '#cc0099'];
    const colorScheme = options?.colorScheme || DEFAULT_COLOR_SCHEME;

    const updatePresences = () => {
        const presences: IPresence[] = Object.values(presence.otherStates);
        onPresencesUpdate(presences, userHostId);
    };

    const submitLocalPresence = () => {
        presence.submitLocalState({
            user,
        });
    };

    submitLocalPresence();

    presence.subscribe().then(() => {
        if (!user.color) {
            user.color = getUnusedColorFromCurrentPresences(colorScheme, presence) || colorScheme[Math.floor(Math.random() * DEFAULT_COLOR_SCHEME.length)];
        }
        submitLocalPresence();
        updatePresences();
    });
    presence.on('add', updatePresences);
    presence.on('update', updatePresences);
    presence.on('remove', updatePresences);
}

function onPresencesUpdate(presences: IPresence[], userHostId: string) {
    const presenceDiv = document.getElementById(userHostId);
    // Clear the current content
    clearUserPresence(userHostId);

    // Iterate over presences and create a circle for each user
    presences.forEach(presence => {
        const user = presence.user;
        const circle = document.createElement('div') as unknown as IDom;
        circle.className = 'presence-item';
        circle.style.backgroundColor = user.color || '#007bff';
        circle.textContent = user.name.charAt(0).toUpperCase();

        // Define event handlers
        const mouseoverHandler = () => {
            circle.style.width = '97px';
            circle.style.padding = "0px 5px";
            circle.textContent = user.name.charAt(0).toUpperCase() + user.name.slice(1);
        };

        const mouseoutHandler = () => {
            circle.style.width = '24px';
            circle.style.padding = "0px";
            circle.textContent = user.name.charAt(0).toUpperCase();
        };

        // Store references to the handlers
        circle._mouseoverHandler = mouseoverHandler;
        circle._mouseoutHandler = mouseoutHandler;

        // Add event listeners
        circle.addEventListener('mouseover', mouseoverHandler);
        circle.addEventListener('mouseout', mouseoutHandler);

        presenceDiv.appendChild(circle);
    });
}

function getUnusedColorFromCurrentPresences(colors: string[], presence: Presence<IPresence>) {
    const usedColors = Object.values(presence.otherStates).map(presence => presence?.user?.color);
    return colors.find(color => !usedColors.includes(color));
}

export function clearUserPresence (userHostId: string) {
    const presenceDiv = document.getElementById(userHostId);
    // Clear the current content
    while (presenceDiv.firstChild) {
        const child = presenceDiv.firstChild as IDom;
        child.removeEventListener('mouseover', child._mouseoverHandler);
        child.removeEventListener('mouseout', child._mouseoutHandler);
        presenceDiv.removeChild(child);
    }
}
