﻿using C1.Win.FlexGrid;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Diagnostics;
using System.Drawing;
using System.IO.IsolatedStorage;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;
using System.Windows.Forms.VisualStyles;

namespace FlexSelection
{
  public partial class FlexGridCalendar : C1FlexGrid
  {
    private int year;
    public FlexGridCalendar()
    {
      InitializeComponent();


      this.AllowSorting = C1.Win.FlexGrid.AllowSortingEnum.None;
      this.AllowEditing = false;


      //Setting Fixed BackColor does not work here...
      //this.Styles.Fixed.BackColor = SystemColors.Control;

      this.Styles.SelectedRowHeader.BackColor = Color.Green;
      this.Styles.SelectedColumnHeader.BackColor = Color.Green;
    }


    public void RenderCalendar (int year)
    {
      this.year = year;
      this.Rows.Count = 13;
      this.Rows.Fixed = 1;
      
      this.Cols.Count = 32;
      this.Cols.Fixed = 1;

      this.Cols[0].Width = 100;
      for (int day = 1; day <= 31; day++)
      {
        this.Cols[day].Width = 25;
      }

      //Header row: day numbers:
      for (int day = 1; day <= 31; day++)
      {
        this[0, day] = day;
      }

      for (int month = 1; month <= 12; month++)
      {
        this[month, 0] = new DateTime(year, month, 1).ToString("MMMM");

        //Days in Month:
        for (int day = 1; day <= DateTime.DaysInMonth(year, month); day++)
        {
          this[month, day] = new DateTime(year, month, day).ToString("ddd");

        }
      }
    }

    /// <summary>
    /// Is the current cell highlighted? The date selection does not match the C1FlexGrid "Selection"!
    /// </summary>
    /// <param name="row"></param>
    /// <param name="col"></param>
    /// <returns></returns>
    public override bool IsCellHighlighted(int row, int col)
    {
      //Get current selection:
      CellRange rangeSel = this.Selection;
      
      if (rangeSel.IsValid == false)
      {
        return base.IsCellHighlighted(row, col);
      }

      DateRange selectedDateRange = this.GetSelectedDateRange();
      
      /*
      Trace.WriteLine("Selection: " + rangeSel + ", Row " + this.Row + "/Col " + this.Col + ", RowSel " + this.RowSel + "/ColSel " + this.ColSel);
      if (selectedDateRange != null)
      {
        Trace.WriteLine( $"\t==>Current Selection: {selectedDateRange}");
      }
      else
      {
        Trace.WriteLine($"\t==>Current Selection: nothing!");
      }*/

      if (selectedDateRange == null)
      {
        //Nothing selected?!
        return base.IsCellHighlighted(row, col);
      }

      //Get Date from cell!
      DateTime? datCell = this.GetDate(row, col);
      //Also get the last day of the month:
      DateTime? lastDayOfMonth = this.GetLastDayOfMonth(row);
      if (datCell != null)
      {
        //Cell has date: is it inside the selection?
        return (selectedDateRange.Start <= datCell && selectedDateRange.End >= datCell);
      }
      else if (lastDayOfMonth != null)
      {
        //Is the current selection starting before the end of the month and ends after? Then select cell:
        return (selectedDateRange.Start <= lastDayOfMonth && selectedDateRange.End >= lastDayOfMonth);
      }
      else
      {
        //Fixed or "out of range" day.
        return false;
      }
    }

    /// <summary>
    /// If style "Styles.SelectedColumnHeader" is set, then all columns whose day is selected shall be rendered as "selected".
    /// 
    /// C1FlexGrid renders only the columns for the "Selection" range.
    /// </summary>
    /// <param name="row"></param>
    /// <param name="col"></param>
    /// <returns></returns>
    public override bool IsCellSelectedColumnHeader(int row, int col)
    {
      //Only for fixed row:
      if (row >= this.Rows.Fixed)
      {
        return base.IsCellSelectedColumnHeader(row, col);
      }

      DateRange selectedDateRange = this.GetSelectedDateRange();
      if (selectedDateRange == null)
      {
        return false;
      }

      //"col" is the day of month. Check whether the selection contains this day.
      if ((selectedDateRange.End.Month - selectedDateRange.Start.Month) > 1)
      {
        //Multi month range:
        return true;
      }
      else if ((selectedDateRange.End.Month - selectedDateRange.Start.Month) == 1)
      {
        //Single month break: 
        return (col >= selectedDateRange.Start.Day || col <= selectedDateRange.End.Day);

      }
      else
      {
        //Inside single month:
        return (col >= selectedDateRange.Start.Day && col <= selectedDateRange.End.Day);
      }
    }

    /// <summary>
    /// After selection change: re-render all affected full rows.
    /// This is necessary, because the selected date ranges do not match the grid rendering range.
    /// </summary>
    /// <param name="e"></param>
    protected override void OnAfterSelChange(RangeEventArgs e)
    {
      //Invalidate cells to the right/left of range:

      CellRange rangeSel = e.NewRange;

      //Invalidate old range:
      if (e.OldRange.IsValid)
      {
        this.Invalidate(e.OldRange.TopRow, 1, e.OldRange.BottomRow, this.Cols.Count - 1);
      }

      //Invalidate new range:
      if (rangeSel.IsValid == true)
      {
        //Invalidate all rows:
        this.Invalidate(rangeSel.TopRow, 1, rangeSel.BottomRow, this.Cols.Count - 1);
      }

      //Invalidate all column headers:
      this.Invalidate(0, this.Cols.Fixed, 0, this.Cols.Count - 1);

      base.OnAfterSelChange(e);
    }

    /// <summary>
    /// Get range of selected dates (start/end range).
    /// </summary>
    /// <remarks>
    /// This is the method that does most of the work: it builds the date range based on the grids "Row/Col" 
    /// (this is the cell where the user started selection) and "RowSel/ColSel" (this is the cell that is currently focused)
    /// 
    /// </remarks>
    /// <returns></returns>
    public DateRange GetSelectedDateRange()
    {
      //We cannot use "CellRange" here, because it seems to be normalized always. Just use it to check whether there
      //is a valid selection.
      CellRange rangeSel = this.Selection;
      if (rangeSel.IsValid == false)
      {
        return null;
      }

      int monthStart;
      int dayStart;
      int monthEnd;
      int dayEnd;

      //"Row" = start row of selection, "RowSel" = current cursor row.

      //Different ways to create a selection by mouse: 
      if (this.RowSel > this.Row && this.ColSel > this.Col)
      {
        //selection down and to the right (default selection):
        monthStart = this.Row;
        dayStart = this.Col;

        monthEnd = this.RowSel;
        dayEnd = this.ColSel;
      }
      else if (this.RowSel > this.Row && this.ColSel < this.Col)
      {
        //Selection down and to the left: 
        monthStart = this.Row;
        dayStart = this.Col;

        monthEnd = this.RowSel;
        dayEnd = this.ColSel;
      }
      else if (this.RowSel <= this.Row && this.ColSel  < this.Col)
      {
        //Move mouse cursor up (or same row) and to the left:
        monthStart = this.RowSel;
        dayStart = this.ColSel;

        monthEnd = this.Row;
        dayEnd = this.Col;
      }
      else if (this.RowSel < this.Row && this.ColSel >= this.Col)
      {
        //Move cursor up and to the right (or same col):
        monthStart = this.RowSel;
        dayStart = this.ColSel;

        monthEnd = this.Row;
        dayEnd = this.Col;
      }
      else
      {
        //Single cell?!
        monthStart = this.Row;
        dayStart = this.Col;

        monthEnd = this.RowSel;
        dayEnd = this.ColSel;
      }
      
      //If "dayStart" is higher than last day of month, then go to first day of following month.
      int daysInMonthStart = DateTime.DaysInMonth(this.year, monthStart);
      if (dayStart > daysInMonthStart)
      {
        ++monthStart;
        dayStart = 1;
        //Can it happen that we move beyond month 12?
        if (monthStart > 12)
        {
          throw new Exception("Invalid: month = " + monthStart);
        }
      }

      DateTime datStart = new DateTime(this.year, monthStart, dayStart);


      //If "dayEnd" is higher than last day of month, then go to last day of this month.
      int daysInMonthEnd = DateTime.DaysInMonth(this.year, monthEnd);
      if (dayEnd > daysInMonthEnd)
      {
        dayEnd = daysInMonthEnd;
      }

      DateTime datEnd = new DateTime(this.year, monthEnd, dayEnd);


      return new DateRange(datStart, datEnd);
    }

    /// <summary>
    /// Get date for cell. Returns null of cell is no valid date (fixed or end of a month with less than 31 days)
    /// </summary>
    /// <param name="row"></param>
    /// <param name="col"></param>
    /// <returns></returns>
    private DateTime? GetDate(int row, int col)
    {
      if (row >= 1 && col >= 1)
      {
        //"row" = month.
        int daysInMonth = DateTime.DaysInMonth(this.year, row);
        if (col > daysInMonth)
        {
          //Month has fewer days: return end of month???? 
          return null;
        }
        else
        {
          return new DateTime(this.year, row, col);
        }
      }
      else
      {
        //Fixed?
        return null;
      }
    }

    /// <summary>
    /// Get the last day of the month. Returns null if row is invalid (fixed)
    /// </summary>
    /// <param name="row"></param>
    /// <returns></returns>
    private DateTime? GetLastDayOfMonth (int row)
    {
      if (row >= 1)
      {
        //"row" = month.
        int daysInMonth = DateTime.DaysInMonth(this.year, row);

        return new DateTime(this.year, row, daysInMonth);
      }
      else
      {
        //Fixed?
        return null;
      }
    }
  }
}
