module wijmo.grid {
    'use strict';

    /**
     * Implements pagination on a @see:FlexGrid control by setting hiding
     * and showing rows using their height property.
     *
     * In most scenarios, pagination is done at the data-layer level, using
     * the grid's @see:FlexGrid.collectionView. But that does not take into
     * account grid-specific operations like expanding and collapsing row
     * groups, so pages with collapsed items appear shorter than the selected
     * page size.
     *
     * The @see:FlexGridPager class addresses this by allowing the grid to
     * load all rows as usual (without data-level pagination), and then
     * hiding rows that are not on the current page by setting their height
     * to zero. This way, when a group is collapsed, additional rows are
     * shown to fill the current page as needed.
     */
    export class FlexGridPager {
        _g: FlexGrid;
        _pageSize: number;
        _pageIndex: number;
        _pageCount: number;
        _firstRow: number;
        _lastRow: number;
        _toRefresh: any;

        /**
         * Creates a new instance of the @see:FlexGridPager class.
         *
         * @param flex The @see:FlexGrid that will be controlled by this @see:FlexGridPager.
         * @param pageSize The number of rows to show on each page.
         */
        constructor(flex: FlexGrid, pageSize: number) {
            this._g = asType(flex, FlexGrid, false);
            this._pageSize = asNumber(pageSize, false, true);
            this._pageIndex = 0;
            flex.loadedRows.addHandler((s, e) => {
                this.invalidate();
            });
            flex.groupCollapsedChanged.addHandler((s, e) => {
                this.refresh();
            });
            this.refresh();

            // refresh after collapse/expand on pivot grid
            if (flex instanceof wijmo.olap.PivotGrid) {
                flex.hostElement.addEventListener('mousedown', (e) => {
                    let icon = closest(e.target, '[wj-pivot-collapse]');
                    if (icon) {
                        this.refresh();
                    }
                });
            }
        }

        // ** object model

        /**
         * Gets the number of rows currently in view.
         */
        get itemCount(): number {
            return this._lastRow - this._firstRow + 1;
        }
        /**
         * Gets the total number of pages.
         */
        get pageCount(): number {
            return this._pageCount;
        }
        /**
         * Gets or sets the index of the current page.
         */
        get pageIndex(): number {
            return this._pageIndex;
        }
        set pageIndex(value: number) {
            this.moveToPage(value);
        }
        /**
         * Gets or sets the page size (number of rows to display on each page).
         */
        get pageSize(): number {
            return this._pageSize;
        }
        set pageSize(value: number) {
            this._pageSize = asNumber(value);
            this.refresh();
        }
        /**
         * Sets the first page as the current page.
         */
        moveToFirstPage(): boolean {
            return this.moveToPage(0);
        }
        /**
         * Sets the last page as the current page.
         */
        moveToLastPage(): boolean {
            return this.moveToPage(this._pageCount - 1);
        }
        /**
         * Sets the next page as the current page.
         */
        moveToNextPage(): boolean {
            return this.moveToPage(this.pageIndex + 1);
        }
        /**
         * Sets the previous page as the current page.
         */
        moveToPreviousPage(): boolean {
            return this.moveToPage(this.pageIndex - 1);
        }
        /**
         * Moves to the page at the specified index.
         */
        moveToPage(index: number): boolean {
            if (index != this._pageIndex) {
                this._pageIndex = asNumber(index);
                this.refresh();
            }
            return this._pageIndex == index;
        }
        /**
         * Schedules a call to the @see:refresh method.
         */
        invalidate() {
            if (this._toRefresh) {
                clearTimeout(this._toRefresh);
            }
            this._toRefresh = setTimeout(() => {
                this._toRefresh = null;
                this.refresh();
            }, 10);
        }
        /**
         * Updates the state of the @see: FlexGridPager.
         */
        refresh() {
            let g = this._g;

            // count non-hidden rows
            let visRows = [];
            for (let i = 0; i < g.rows.length; i++) {
                let r = g.rows[i];
                if (r.visible && !r._getFlag(RowColFlags.ParentCollapsed)) {
                    visRows.push(r);
                }
            }

            // update page count, row indices
            this._pageCount = 1;
            this._firstRow = 0;
            this._lastRow = visRows.length ? visRows.length - 1 : 0;
            if (this._pageSize > 0 && visRows.length) {

                // compute page count and index
                this._pageCount = this._pageSize ? Math.ceil(visRows.length / this._pageSize) : 1;
                this._pageIndex = Math.max(0, Math.min(this._pageCount, this._pageIndex))

                // compute row range for current page
                let start = Math.min(this._pageIndex * this._pageSize, visRows.length - 1);
                this._firstRow = visRows[start].index;
                this._lastRow = visRows[Math.min(start + this._pageSize, visRows.length) - 1].index;
            }

            // show/hide rows
            g.deferUpdate(() => {
                for (let i = 0; i < g.rows.length; i++) {
                    g.rows[i].height = (i < this._firstRow || i > this._lastRow)
                        ? 0
                        : null;
                }
            });

            // all done
            this.onStateChanged(EventArgs.empty);
        }

        /**
         * Occurs when the page state changes.
         */
        readonly stateChanged = new Event();
        /**
         * Raises the @see:stateChanged event.
         */
        onStateChanged(e?: EventArgs) {
            this.stateChanged.raise(this, e);
        }
    }
}