﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Xml;
using Newtonsoft.Json;

namespace SharedDataSetReferenceUpdater
{
	public class SharedDataSetReferenceUpdater
	{
		private const string START = "server:";
		private static string ns;
		private static XmlNamespaceManager nsmgr;
		private static Dictionary<string, ServerInfo> serverMap;

		class ServerInfo
		{
			public string login;
			public string password;
			public string authToken;
			public Dictionary<string, string> datasets;
			public Dictionary<string, string> images;
			public Dictionary<string, string> styleSheets;
		}

		class DataSetInfo
		{
			[JsonProperty("Name")]
			public string Name { get; set; }

			[JsonProperty("_id")]
			public string Id { get; set; }
		}

		class ImageInfo
		{
			[JsonProperty("Name")]
			public string Name { get; set; }

			[JsonProperty("_id")]
			public string Id { get; set; }
		}

		class StyleSheetInfo
		{
			[JsonProperty("Name")]
			public string Name { get; set; }

			[JsonProperty("_id")]
			public string Id { get; set; }
		}

		static SharedDataSetReferenceUpdater()
		{
			serverMap = new Dictionary<string, ServerInfo>();
		}

		private static ServerInfo ReceiveServerInfo(string serverUri)
		{
			var serverInfo = new ServerInfo();

			Console.WriteLine("Enter login for " + serverUri);
			serverInfo.login = Console.ReadLine();
			Console.WriteLine("Enter password for " + serverUri);
			serverInfo.password = Console.ReadLine();

			var authToken = ReceiveAuthToken(serverUri, serverInfo);
			if (authToken != null)
			{
				serverInfo.authToken = authToken;
				ReceiveEntitiesLists(serverUri, serverInfo);
				serverMap.Add(serverUri, serverInfo);
				return serverInfo;
			}
			throw new Exception("Authorization failure");
		}

		private static void PrepareXmlNamespaceManager(XmlDocument report)
		{
			var xmlns = report.DocumentElement.Attributes["xmlns"].Value;
			nsmgr = new XmlNamespaceManager(report.NameTable);

			ns = "namespace";
			nsmgr.AddNamespace(ns, xmlns);
			ns = ns + ":";
		}

		private static string ReceiveAuthToken(string serverUri, ServerInfo serverInfo)
		{
			var client = new HttpClient();
			client.BaseAddress = new Uri(serverUri);
			client.DefaultRequestHeaders.Accept.Add(new System.Net.Http.Headers.MediaTypeWithQualityHeaderValue("application/json"));
			var content = new StringContent("{\"User\": \"" + serverInfo.login + "\",\"Password\": \"" + serverInfo.password + "\"}");

			try
			{
				dynamic resultObject = JsonConvert.DeserializeObject(client.PostAsync("/api/accounts/login", content).Result.Content.ReadAsStringAsync().Result);
				return resultObject.Token.ToString();
			}
			catch (Exception ex)
			{
				Console.WriteLine("Error on login attempt: " + ex.Message);
				return null;
			}
		}

		public static XmlDocument UpdateSharedResourceReferences(XmlDocument report)
		{
			PrepareXmlNamespaceManager(report);
			var serverUri = FindServerUri(report);
			if (string.IsNullOrEmpty(serverUri))
				return null;

			var serverInfo = new ServerInfo();
			if (!serverMap.TryGetValue(serverUri, out serverInfo))
			{
				serverInfo = ReceiveServerInfo(serverUri);
			}

			UpdateSharedDatasetReferences(report, serverUri);
			UpdateSharedImageReferences(report, serverUri);
			UpdateSharedStyleSheetReferences(report, serverUri);

			return report;
		}

		private static void UpdateSharedDatasetReferences(XmlDocument report, string serverUri)
		{
			try
			{
				var sharedDSList = report.GetElementsByTagName("SharedDataSetReference");

				for (int i = 0; i < sharedDSList.Count; i++)
				{
					var datasetRef = sharedDSList[i].InnerText;
					if (!datasetRef.StartsWith(START))
					{
						Console.WriteLine("Incorrect <SharedDataSetReference> inner text");
						continue;
					}
					datasetRef = datasetRef.Remove(0, 7);
					if (datasetRef.StartsWith("///datasets/"))
						continue;

					var id = FindDataSetId(datasetRef, serverUri);
					if (id == null)
					{
						Console.WriteLine(string.Format("No dataset by name {0} found", datasetRef));
						continue;
					}

					sharedDSList[i].InnerText = string.Format("{0}{1}{2}", START, "///datasets/", id);
				}
			}
			catch (Exception e)
			{
				Console.WriteLine(e.Message);
			}
		}

		private static void UpdateSharedImageReferences(XmlDocument report, string serverUri)
		{
			try
			{
				var sharedImageList = report.GetElementsByTagName("Image");

				for (int i = 0; i < sharedImageList.Count; i++)
				{
					var imageRef = sharedImageList[i]["Value"].InnerText;
					if (!imageRef.StartsWith(START))
					{
						Console.WriteLine("Incorrect SharedImageReference inner text");
						continue;
					}
					imageRef = imageRef.Remove(0, 7);
					if (imageRef.StartsWith("///images/"))
						continue;

					if (!imageRef.StartsWith("//images/"))
					{
						Console.WriteLine("Incorrect SharedImageReference inner text");
						continue;
					}
					imageRef = imageRef.Remove(0, 9);
					var id = FindImageId(imageRef, serverUri);
					if (id == null)
					{
						Console.WriteLine(string.Format("No image by name {0} found", imageRef));
						continue;
					}

					sharedImageList[i]["Value"].InnerText = string.Format("{0}{1}{2}", START, "///images/", id);
				}
			}
			catch (Exception e)
			{
				Console.WriteLine(e.Message);
			}
		}

		private static void UpdateSharedStyleSheetReferences(XmlDocument report, string serverUri)
		{
			try
			{
				var sharedStyleSheetList = report.GetElementsByTagName("StyleSheetValue");

				for (int i = 0; i < sharedStyleSheetList.Count; i++)
				{
					var styleSheetRef = sharedStyleSheetList[i].InnerText;
					if (!styleSheetRef.StartsWith(START))
					{
						Console.WriteLine("Incorrect SharedStyleSheetReference inner text");
						continue;
					}
					styleSheetRef = styleSheetRef.Remove(0, 7);
					if (styleSheetRef.StartsWith("///stylesheets/"))
						continue;

					if (!styleSheetRef.StartsWith("//stylesheets/"))
					{
						Console.WriteLine("Incorrect SharedStyleSheetReference inner text");
						continue;
					}
					styleSheetRef = styleSheetRef.Remove(0, 14);
					var id = FindStyleSheetId(styleSheetRef, serverUri);
					if (id == null)
					{
						Console.WriteLine(string.Format("No image by name {0} found", styleSheetRef));
						continue;
					}

					sharedStyleSheetList[i].InnerText = string.Format("{0}{1}{2}", START, "///images/", id);
				}
			}
			catch (Exception e)
			{
				Console.WriteLine(e.Message);
			}
		}

		private static string FindServerUri(XmlDocument report)
		{
			var xmlns = report.DocumentElement.Attributes["xmlns"].Value;
			var nsmgr = new XmlNamespaceManager(report.NameTable);

			var ns = "namespace";
			nsmgr.AddNamespace(ns, xmlns);
			ns = ns + ":";

			var customProperties = report.SelectNodes(ns + "Report/" + ns + "CustomProperties/" + ns + "CustomProperty", nsmgr);
			return (from XmlNode customProperty in customProperties 
					where customProperty["Name"].InnerText == "ReportServerUri" 
					select customProperty["Value"].InnerText)
					.FirstOrDefault();
		}

		private static string FindDataSetId(string datasetName, string serverUri)
		{
			try
			{
				ServerInfo serverInfo;
				if (!serverMap.TryGetValue(serverUri, out serverInfo))
				{
					serverInfo = ReceiveServerInfo(serverUri);
				}

				var dataSetId = serverInfo.datasets[datasetName];
				if (dataSetId == null)
				{
					Console.WriteLine("There is no dataset " + datasetName);
					return null;
				}
				return dataSetId;
			}
			catch (Exception ex)
			{
				Console.WriteLine("Error on retrieving dataset info: " + ex.Message);
				return null;
			}
		}

		private static string FindImageId(string imageName, string serverUri)
		{
			try
			{
				ServerInfo serverInfo;
				if (!serverMap.TryGetValue(serverUri, out serverInfo))
				{
					serverInfo = ReceiveServerInfo(serverUri);
				}

				var imageId = serverInfo.images[imageName];
				if (imageId == null)
				{
					Console.WriteLine("There is no image " + imageName);
					return null;
				}
				return imageId;
			}
			catch (Exception ex)
			{
				Console.WriteLine("Error on retrieving image info: " + ex.Message);
				return null;
			}
		}

		private static string FindStyleSheetId(string styleSheetName, string serverUri)
		{
			try
			{
				ServerInfo serverInfo;
				if (!serverMap.TryGetValue(serverUri, out serverInfo))
				{
					serverInfo = ReceiveServerInfo(serverUri);
				}

				var styleSheetId = serverInfo.styleSheets[styleSheetName];
				if (styleSheetId == null)
				{
					Console.WriteLine("There is no styleSheet " + styleSheetName);
					return null;
				}
				return styleSheetId;
			}
			catch (Exception ex)
			{
				Console.WriteLine("Error on retrieving styleSheet info: " + ex.Message);
				return null;
			}
		}

		private static void ReceiveEntitiesLists(string serverUri, ServerInfo serverInfo)
		{
			var client = new HttpClient();
			client.BaseAddress = new Uri(serverUri);
			client.DefaultRequestHeaders.Accept.Add(new System.Net.Http.Headers.MediaTypeWithQualityHeaderValue("application/json"));
			client.DefaultRequestHeaders.Add("AuthToken", serverInfo.authToken);

			var datasetsObject = JsonConvert.DeserializeObject<List<DataSetInfo>>(client.GetAsync("/api/datasets").Result.Content.ReadAsStringAsync().Result);
			serverInfo.datasets = new Dictionary<string, string>();
			foreach (var dataset in datasetsObject)
			{
				serverInfo.datasets.Add(dataset.Name, dataset.Id);
			}

			var imagesObject = JsonConvert.DeserializeObject<List<ImageInfo>>(client.GetAsync("/api/images").Result.Content.ReadAsStringAsync().Result);
			serverInfo.images = new Dictionary<string, string>();
			foreach (var image in imagesObject)
			{
				serverInfo.images.Add(image.Name, image.Id);
			}

			var stylesheetsObject = JsonConvert.DeserializeObject<List<ImageInfo>>(client.GetAsync("/api/stylesheets").Result.Content.ReadAsStringAsync().Result);
			serverInfo.styleSheets = new Dictionary<string, string>();
			foreach (var stylesheet in stylesheetsObject)
			{
				serverInfo.styleSheets.Add(stylesheet.Name, stylesheet.Id);
			}
		}
	}
}
