import {
  GridUpdate, GridRowCountUpdateAction, ActionTypeCase, GridAction,
  GridViewportOffsetAction, GridRowUpdateAction, GridViewportRequest, GridSetFilterRequest
} from '../../messages/grid';
import * as _ from 'lodash';
import 'wijmo';
import {RemoteDispatcher} from '../connect';
import NotifyCollectionChangedEventArgs = wijmo.collections.NotifyCollectionChangedEventArgs;
import {createRandomString} from '../uuid';
import {Randomizer} from '../Randomizer';

export class GridViewModel extends wijmo.collections.CollectionView {
  private rowCount: number = 0;
  private startIndex: number = 0;
  private endIndex: number = 0;

  private readonly data: any[];
  private readonly bufferSize: number;
  private readonly loadCount: number;

  constructor(private dispatcher: RemoteDispatcher) {
    super();

    this.bufferSize = 10;
    this.loadCount = this.bufferSize * 2 + 30;
    this.data = [];
    this.sourceCollection = this.data;
    this.canFilter = false;
    this.canSort = false;

    // this.canChangePage = true;
    // this.sortDescriptions.collectionChanged.addHandler(() => {
    //   this.requestViewport();
    // });

    // this.dispatcher.subscribe(
    //   new GridViewportRequest(),
    //   (envelope: Envelope) => {
    //     let update = envelope.getGridUpdate();
    //     this.onGridUpdateMessage(update);
    //   });

    this.requestViewport();
  }

   public get totalItemCount() {
    return this.rowCount;
   }

  public setFilter(filterTarget?: string): void {
    if (_.isUndefined(filterTarget)) {
      return;
    }
    const setFilterRequest = new GridSetFilterRequest()
      .setTarget(filterTarget);
    this.dispatcher.send(setFilterRequest);
  }

  public setViewport(start: number, end: number): void {
    // invalid constraint
    if (start > end) {
      return;
    }

    // no need to load data if requested range is already within
    // last loaded viewport
    const isStartLoaded = _.inRange(start, this.startIndex, this.endIndex);
    const isEndLoaded = _.inRange(end, this.startIndex, this.endIndex + 1);
    if (isStartLoaded && isEndLoaded) {
      return;
    }

    let loadStart: number; let loadEnd: number;
    let deleteStart: number; let deleteEnd: number;
    if (!isStartLoaded && !isEndLoaded) {
      deleteStart = this.startIndex;
      deleteEnd = this.endIndex;
      loadStart = Math.max(0, start - this.bufferSize);
      loadEnd = Math.min(this.rowCount, end + this.bufferSize);
    } else if (!isStartLoaded) {
      deleteStart = end + this.bufferSize + 1;
      deleteEnd = this.endIndex;
      loadStart = Math.max(0, start - this.bufferSize);
      loadEnd = this.startIndex;
    } else {
      deleteStart = this.startIndex;
      deleteEnd = Math.max(0, start - this.bufferSize);
      loadStart = this.endIndex;
      loadEnd = Math.min(this.rowCount, end + this.bufferSize);
    }

    this.deferUpdate(() => this.deleteEntries(deleteStart, deleteEnd));
    this.startIndex = Math.max(0, start - this.bufferSize);
    this.endIndex = Math.min(this.rowCount, end + this.bufferSize);
    this.requestViewport(loadStart, loadEnd);
  }

  private setRowCount(value: number): void {
    this.rowCount = value;
    this.data.length = value;
    this.startIndex = 0;
    this.endIndex = Math.min(this.rowCount, this.loadCount);
  }

  private deleteEntries(startIndex: number, endInclusiveIndex: number): void {
    for (let i = startIndex; i <= endInclusiveIndex; i++) {
      delete this.data[i];
    }
    this.itemsRemoved.clear();
  }

  private requestViewport(start?: number, end?: number): void {
    // let request = new GridViewportRequest();
    if (!_.isUndefined(start) && !_.isUndefined(end)) {
      // request = request
      //   .setStartIndex(start)
      //   .setEndIndex(end);
      setTimeout(() => this.generate(start, end), 500);
    } else {
      this.init();
    }
    // this.dispatcher.send(request);
  }

  private onGridUpdateMessage(message: GridUpdate): void {
    this.beginUpdate();
    let actions: GridAction[] = message.getActionsList();
    let length: number = actions.length;
    for (let i: number = 0; i < length; i++) {
      const action = actions[i];
      let actionTypeCase: ActionTypeCase = action.getActionTypeCase();
      switch (actionTypeCase) {
        case ActionTypeCase.ROW_COUNT_UPDATE:
          const setRowCountAction: GridRowCountUpdateAction = action.getRowCountUpdate();
          this.setRowCount(setRowCountAction.getRowCount());
          break;
        case ActionTypeCase.VIEWPORT_OFFSET:
          const moveViewportAction: GridViewportOffsetAction = action.getViewportOffset();
          const offset: number = moveViewportAction.getOffset();
          this.moveCurrentToPosition(offset);
          break;
        case ActionTypeCase.ROW_UPDATE:
          const updateRowAction: GridRowUpdateAction = action.getRowUpdate();
          this.createRow(updateRowAction, this.data);
          break;
        default:
          throw new Error('Unknown grid action type: ' + actionTypeCase);
      }
    }
    this.endUpdate();
  }

  private createRow(updateRowAction: GridRowUpdateAction, container: any[]): void {
    const index: number = updateRowAction.getIndex();
    const data: Map<string, string> = updateRowAction.getDataMap();
    let datum = container[index];
    if (_.isNil(datum)) {
      datum = {
        children: []
      };
      container[index] = datum;
    }
    data.forEach((value, columnName) => {
      datum[columnName] = value;
    });

    let items = datum.children;
    const children = updateRowAction.getChildRowsList();
    if (children && children.length > 0) {
      for (let i = 0; i < children.length; i++) {
        this.createRow(children[i], items);
      }
    }
  }

  private generate(start: number, end: number) {
    let update: GridUpdate  = new GridUpdate();
    let actions = [];
    for (let i = start; i < end; i++) {
      let rowUpdate = new GridRowUpdateAction();
      rowUpdate.setIndex(i);
      const dataMap = rowUpdate.getDataMap();
      GridViewModel.generateData('parent-' + i, dataMap);

      let childCount = Randomizer.getNextInt(1, 2, true);
      let childList = [];
      for (let j = 0; j < childCount; j++) {
        let childUpdate = new GridRowUpdateAction();
        childUpdate.setIndex(j);
        GridViewModel.generateData('child-' + j, childUpdate.getDataMap());
        childList.push(childUpdate);
      }
      rowUpdate.setChildRowsList(childList);

      let action = new GridAction();
      action.setRowUpdate(rowUpdate);
      actions.push(action);
    }
    update.setActionsList(actions);

    this.onGridUpdateMessage(update);
  }

  private init() {
    let update: GridUpdate  = new GridUpdate();
    let action = new GridAction();
    let rowUpdate = new GridRowCountUpdateAction();
    rowUpdate.setRowCount(10000);
    action.setRowCountUpdate(rowUpdate);
    update.addActions(action);
    this.onGridUpdateMessage(update);
    this.generate(0, this.loadCount);
  }

  private static generateData(id: string, map: Map<string, string>) {
    map.set('group_groupId', id);
    map.set('group_name', createRandomString(5));
    map.set('client', createRandomString(5));
    map.set('state', createRandomString(5));
    map.set('group_orderCount', createRandomString(5));
    map.set('group_totalQty', createRandomString(5));
    map.set('group_buyQty', createRandomString(5));
    map.set('group_sellQty', createRandomString(5));
    map.set('group_totalNotional', createRandomString(5));
    map.set('group_buyNotional', createRandomString(5));
    map.set('group_sellNotional', createRandomString(5));
    map.set('group_workedQty', createRandomString(5));
    map.set('group_unworkedQty', createRandomString(5));
    map.set('group_execQty', createRandomString(5));
    map.set('group_leavesQty', createRandomString(5));
    map.set('group_workedPct', createRandomString(5));
    map.set('group_unworkedPct', createRandomString(5));
    map.set('group_execPct', createRandomString(5));
    map.set('group_workedNotional', createRandomString(5));
    map.set('group_execNotional', createRandomString(5));
    map.set('group_unworkedNotional', createRandomString(5));
    map.set('group_leavesNotional', createRandomString(5));
    map.set('trader', createRandomString(5));
    map.set('text', createRandomString(5));
  }
}


/*
 _storeItems = function(n, t) {
   var r, i;
   if (this._refresh || this._data.length != this.totalItemCount) {
     for (this._data.length = this.totalItemCount, i = 0; i < this._data.length; i++) this._data[i] = null;
     this._refresh = !1
   }
   for (t || (this._loadOffset = 0), r = this._loadOffset + (this._skip || 0), i = 0; i < n.length; i++) {
    this._data[i + r] = n[i];
   }
   this._loadOffset += n.length
 }
 _performSetWindow = function(t, i) {
   var e, u, f, r;
   for (t = n.asInt(t), i = n.asInt(i), n.assert(i >= t, 'Start must be smaller than end.'), e = n.isNumber(this._start) && t > this._start, this._start = t, this._end = i, u = !1, r = t; r < i && r < this._data.length && !u; r++) u = this._data[r] == null;
   if (u) {
     for (f = Math.max(0, e ? t : i - this.pageSize), r = f; r < this._data.length && this._data[r] != null; r++) {
      f++;
     }
     this._skip = f;
     this._getData()
   }
 }
* */
